from django.shortcuts import render, redirect
from django.views.generic import FormView, ListView,TemplateView,DetailView, View
from django.urls import reverse, reverse_lazy
from .models import BoardMember, AdvisoryMember,NewsArticle, Document, MediaItem, TeamMember, Department
from django.shortcuts import get_object_or_404
from .forms import VolunteerApplicationForm, InternshipApplicationForm, JobApplicationForm, ContactForm
from django.contrib import messages
import os
from django.conf import settings
from django.http import FileResponse, Http404



class AboutView(TemplateView):
    template_name = 'main/about.html'
    success_url = reverse_lazy('about')
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['board_members'] = BoardMember.objects.all()
        context['advisory_members'] = AdvisoryMember.objects.all()
        context['team_members'] = TeamMember.objects.all()
        return context
    
class CareersView(TemplateView):
    template_name = "main/careers.html"

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['volunteer_form'] = VolunteerApplicationForm()
        context['internship_form'] = InternshipApplicationForm()
        context['job_form'] = JobApplicationForm()
        return context

    def post(self, request, *args, **kwargs):
        form_type = request.POST.get('form_type')

        if form_type == 'volunteer':
            form = VolunteerApplicationForm(request.POST)
        elif form_type == 'internship':
            form = InternshipApplicationForm(request.POST, request.FILES)
        elif form_type == 'job':
            form = JobApplicationForm(request.POST, request.FILES)
        else:
            return redirect('careers')

        if form.is_valid():
            form.save()
            return redirect('careers')  # You can customize this to show a thank-you message
        else:
            context = self.get_context_data()
            if form_type == 'volunteer':
                context['volunteer_form'] = form
            elif form_type == 'internship':
                context['internship_form'] = form
            elif form_type == 'job':
                context['job_form'] = form
            return self.render_to_response(context)
        
class HomepageView(TemplateView):
    template_name = 'main/home.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['news'] = NewsArticle.objects.order_by('-published_at')[:3]
        return context
    
class NewsListView(ListView):
    model = NewsArticle
    template_name = 'main/news_list.html'
    context_object_name = 'news'
    paginate_by = 9
    
    def get_queryset(self):
        return NewsArticle.objects.filter(is_published=True)

class NewsDetailView(DetailView):
    model = NewsArticle
    template_name = 'main/news_detail.html'
    context_object_name = 'article'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        
        # Get related articles
        context['related_articles'] = NewsArticle.objects.filter(
            is_published=True
        ).exclude(pk=self.object.pk)[:3]
        
        # Generate current URL - this works in both development and production
        context['current_url'] = self.request.build_absolute_uri()
        
        # Alternative method if you want to force a specific domain
        # if settings.DEBUG:
        #     # For development
        #     context['current_url'] = f"http://127.0.0.1:8000{self.request.path}"
        # else:
        #     # For production
        #     context['current_url'] = f"https://yourdomain.com{self.request.path}"
        
        # Page metadata
        context['page_title'] = self.object.title
        context['page_description'] = getattr(self.object, 'summary', self.object.title)
        
        # Image URL
        if hasattr(self.object, 'image') and self.object.image:
            context['page_image'] = self.request.build_absolute_uri(self.object.image.url)
        
        # Debug information (remove in production)
        if settings.DEBUG:
            context['debug_info'] = {
                'current_url': context['current_url'],
                'request_path': self.request.path,
                'request_method': self.request.method,
                'request_host': self.request.get_host(),
            }
        
        return context

class ResourcesView(TemplateView):
    template_name = 'main/resources.html'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['organizational_policies'] = Document.objects.filter(category='organizational')
        context['program_policies'] = Document.objects.filter(category='program')
        context['strategic_plans'] = Document.objects.filter(category='strategic')
        context['annual_reports'] = Document.objects.filter(category='report')

        
        return context
    
class ContactView(FormView):
    template_name = 'main/contact.html'
    form_class = ContactForm
    success_url = reverse_lazy('contact')
    
    def form_valid(self, form):
        # Process the form data
        form.save()
        messages.success(self.request, 'Thank you for contacting us! We will get back to you soon.')
        return super().form_valid(form)
    
class BoardProfilesView(View):
    def get(self, request):
        return render(request, 'main/board_profiles.html')
    

class DownloadBoardProfilesPDF(View):
    def get(self, request):
        pdf_path = os.path.join(settings.MEDIA_ROOT, 'docs/board_profiles.pdf')
        print(f"DEBUG: Looking for file at {pdf_path}")
        if os.path.exists(pdf_path):
            return FileResponse(open(pdf_path, 'rb'), as_attachment=True, filename='board_profiles.pdf')
        else:
            raise Http404("The Board Profiles PDF could not be found.")
    
    
class MediaPageView(ListView):
    model = MediaItem
    template_name = 'main/media.html'
    context_object_name = 'media_items'
    
    def get_queryset(self):
        return MediaItem.objects.all()
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['media_items'] = self.get_queryset()
        return context
class MediaDetailView(DetailView):
    model = MediaItem
    template_name = 'main/media_detail.html'
    context_object_name = 'media_item'
    
    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)
        context['media_item'] = self.get_object()
        return context

class EducationView(TemplateView):
    template_name = 'main/education.html'



class LivelihoodView(TemplateView):
    template_name = 'main/livelihood.html'
    
    
class WashView(TemplateView):
    template_name = 'main/wash.html'
    
class SocialJusticeView(TemplateView):
    template_name = 'main/social-justice.html'
    
class ClimateView(TemplateView):
    template_name = 'main/climate.html'
    
class EcoTourismView(TemplateView):
    template_name = 'main/ecotourism.html'
    
class RestorationView(TemplateView):
    template_name = 'main/restoration.html'
    
class AgribusinessView(TemplateView):
    template_name = 'main/agribusiness.html'
    
    
class FinancialLiteracyView(TemplateView):
    template_name = 'main/financial-literacy.html'


class TeamPageView(TemplateView):
    template_name = "main/teams.html"

    def get_context_data(self, **kwargs):
        context = super().get_context_data(**kwargs)

        context["page_title"] = "Meet Our Team"
        context["page_subtitle"] = "Passionate professionals dedicated to excellence"
        context["leadership_section_title"] = "Leadership"
        context["team_section_title"] = "Our Team"

        context["leadership_team"] = TeamMember.objects.filter(is_leadership=True)
        context["team_members"] = TeamMember.objects.filter(is_leadership=False)
        context["departments"] = Department.objects.all()

        return context